# -*- coding: utf-8 -*-
"""
VoicePad Voice Controller - OPTIMIZED

SIMPLE voice commands:
- "Dictation start" -> Start recording
- "Dictation stop" -> Stop recording
- "Dictation exit" -> Close app
- "Dictation dark/light/frost" -> Change theme

Uses VOSK for 100% offline, always-on listening.
Plays BEEP confirmation so user knows it worked (even away from PC).

OPTIMIZATIONS APPLIED:
1. LIMITED GRAMMAR mode - only recognizes specific phrases (95%+ accuracy)
2. EndpointerMode.SHORT - 25% faster response
3. Custom endpointer delays - 0.3s after speech (vs default 1.0s)
4. MaxAlternatives(3) - checks multiple hypotheses for accent tolerance
5. SetPartialWords(False) - faster single-phrase detection

REQUIREMENTS:
    pip install vosk pyaudio

VOSK MODEL (ENGLISH ONLY):
    Download: https://alphacephei.com/vosk/models/vosk-model-small-en-us-0.15.zip
    Extract to: models/vosk-model-small-en-us-0.15/
"""

import json
import logging
import threading
import time
from pathlib import Path
from typing import Callable, Optional

from .beep import (
    play_error_beep,
    play_ready_beep,
    play_start_beep,
    play_stop_beep,
)

log = logging.getLogger("voicepad.voice_control")

# ═══════════════════════════════════════════════════════════════════════════════
# CONFIGURATION
# ═══════════════════════════════════════════════════════════════════════════════

# Audio settings (MUST be exactly these for VOSK)
SAMPLE_RATE = 16000
CHANNELS = 1
CHUNK_SIZE = 8000  # 250ms of audio at 16kHz - more context = better recognition

# MINIMAL GRAMMAR - Small model needs FEWER phrases for accuracy
# Phrases for focused recognition with strict command matching
# Wake word: "DICTATION" (single word, cleaner than "voice pad")
# NOTE: Only "dictation" is accepted - "dictate" was removed to prevent false triggers
VOICE_CONTROL_GRAMMAR = [
    # START - begin recording
    "dictation start",
    "dictation",
    # STOP - stop recording
    "dictation stop",
    # EXIT - close app
    "dictation exit",
    # THEMES
    "dictation dark",
    "dictation light",
    "dictation frost",
    # Catch-all
    "[unk]",
]

class VoiceController:
    """
    Hands-free voice control for VoicePad.

    COMMANDS (say these):
        "Dictation start" -> on_start_recording()
        "Dictation stop"  -> on_stop_recording()
        "Dictation exit"  -> on_close_app()
        "Dictation dark"  -> on_theme_dark()
        "Dictation light" -> on_theme_light()
        "Dictation frost" -> on_theme_frost()

    NOTE: Wake word is "dictation" (single word, cleaner recognition)

    USAGE:
        controller = VoiceController(
            on_start_recording=my_start_func,
            on_stop_recording=my_stop_func,
        )
        controller.start()  # Begin listening
        # ... later ...
        controller.stop()   # Stop listening
    """

    def __init__(
        self,
        on_start_recording: Optional[Callable] = None,
        on_stop_recording: Optional[Callable] = None,
        on_close_app: Optional[Callable] = None,
        on_theme_dark: Optional[Callable] = None,
        on_theme_light: Optional[Callable] = None,
        on_theme_frost: Optional[Callable] = None,
        vosk_model_path: Optional[str] = None,
    ):
        """
        Initialize voice controller. ENGLISH ONLY.

        Args:
            on_start_recording: Callback when "Dictation start" is heard
            on_stop_recording: Callback when "Dictation stop" is heard
            on_close_app: Callback when "Dictation exit" is heard
            on_theme_dark: Callback when "Dictation dark" is heard
            on_theme_light: Callback when "Dictation light" is heard
            on_theme_frost: Callback when "Dictation frost" is heard
            vosk_model_path: Path to VOSK model (auto-detect if None)
        """
        self.on_start_recording = on_start_recording
        self.on_stop_recording = on_stop_recording
        self.on_close_app = on_close_app
        self.on_theme_dark = on_theme_dark
        self.on_theme_light = on_theme_light
        self.on_theme_frost = on_theme_frost

        # State
        self._running = False
        self._thread: Optional[threading.Thread] = None
        self._vosk_model = None
        self._vosk_recognizer = None
        self._audio_stream = None
        self._pyaudio = None

        # Debounce to prevent double-triggers
        self._last_command_time = 0
        self._command_cooldown = 0.8  # seconds - prevents rapid re-triggers
        self._last_executed_text = ""  # Track last executed command text
        self._last_executed_time = 0  # When it was executed

        # Find VOSK model
        self._model_path = vosk_model_path or self._find_vosk_model()

        # Stats
        self.stats = {
            "commands_heard": 0,
            "start_commands": 0,
            "stop_commands": 0,
            "close_commands": 0,
            "theme_commands": 0,
            "start_time": None,
        }

        log.info("VoiceController initialized (optimized mode)")

    def _find_vosk_model(self) -> Optional[str]:
        """
        Auto-detect VOSK model - same pattern as Whisper model detection.
        Checks bundled models first (in _internal for PyInstaller or project root).
        """
        import sys

        # Model names to search for (ENGLISH ONLY)
        model_names = [
            "vosk-model-small-en-us-0.15",  # English small (40MB) - lightweight & fast
        ]

        # Build search paths - same pattern as Whisper bundled models
        search_paths = []

        # 1. BUNDLED: When running as compiled EXE (PyInstaller)
        if getattr(sys, "frozen", False):
            app_dir = Path(sys.executable).parent
            # PyInstaller puts data in _internal folder
            search_paths.append(app_dir / "_internal" / "models")
            search_paths.append(app_dir / "models")

        # 2. DEVELOPMENT: Project root models folder
        project_root = Path(__file__).parent.parent.parent
        search_paths.append(project_root / "models")

        # 3. FALLBACK: Common user locations
        search_paths.extend(
            [
                Path("models"),
                Path.home() / "vosk-models",
                Path.home() / ".vosk" / "models",
            ]
        )

        # Search for model
        for base in search_paths:
            if not base.exists():
                continue

            for model_name in model_names:
                model_path = base / model_name

                # Check for valid VOSK model (must have conf/mfcc.conf)
                if model_path.exists():
                    conf_file = model_path / "conf" / "mfcc.conf"
                    am_file = model_path / "am" / "final.mdl"

                    if conf_file.exists() or am_file.exists():
                        log.info(f"Found bundled VOSK model: {model_path}")
                        return str(model_path)

        log.warning("No VOSK model found. Voice control unavailable.")
        log.warning("Run download_vosk_model.bat to install the model.")
        return None

    def _init_vosk(self) -> bool:
        """
        Initialize VOSK recognizer with grammar mode.

        NOTE: Some optimizations (EndpointerMode, SetEndpointerDelays)
        may not be available in all VOSK versions. We try them but
        fall back gracefully.
        """
        if not self._model_path:
            log.error("No VOSK model path - cannot initialize")
            return False

        try:
            import vosk

            vosk.SetLogLevel(-1)  # Suppress verbose logs

            log.info(f"Loading VOSK model from: {self._model_path}")
            self._vosk_model = vosk.Model(self._model_path)

            # Create recognizer with LIMITED vocabulary grammar
            grammar_json = json.dumps(VOICE_CONTROL_GRAMMAR)
            self._vosk_recognizer = vosk.KaldiRecognizer(self._vosk_model, SAMPLE_RATE, grammar_json)

            # ══════════════════════════════════════════════════════════════════
            # OPTIONAL SPEED OPTIMIZATIONS (may not exist in all VOSK versions)
            # ══════════════════════════════════════════════════════════════════

            # Try EndpointerMode.SHORT (25% faster)
            try:
                from vosk import EndpointerMode

                self._vosk_recognizer.SetEndpointerMode(EndpointerMode.SHORT)
                log.info("EndpointerMode.SHORT enabled")
            except (ImportError, AttributeError) as e:
                log.debug(f"EndpointerMode not available: {e}")

            # Try custom endpointer delays
            try:
                self._vosk_recognizer.SetEndpointerDelays(3.0, 0.3, 10.0)
                log.info("Custom endpointer delays set (0.3s after speech)")
            except AttributeError:
                log.debug("SetEndpointerDelays not available")

            # DON'T use MaxAlternatives - it changes result format and
            # removes confidence scores. Simple mode works better.
            # self._vosk_recognizer.SetMaxAlternatives(3)

            # Enable word-level detection for better command matching
            try:
                self._vosk_recognizer.SetWords(True)
                log.info("Word-level detection enabled")
            except AttributeError:
                log.debug("SetWords not available")

            log.info(f"VOSK initialized: grammar={len(VOICE_CONTROL_GRAMMAR)} phrases")
            return True

        except ImportError:
            log.error("VOSK not installed. Run: pip install vosk")
            return False
        except Exception as e:
            log.error(f"VOSK initialization failed: {e}")
            return False

    def _init_audio(self) -> bool:
        """Initialize audio capture with correct settings for VOSK."""
        try:
            import pyaudio

            self._pyaudio = pyaudio.PyAudio()

            # Open microphone stream - MUST match VOSK requirements
            self._audio_stream = self._pyaudio.open(
                format=pyaudio.paInt16,  # 16-bit PCM
                channels=CHANNELS,  # Mono
                rate=SAMPLE_RATE,  # 16kHz
                input=True,
                frames_per_buffer=CHUNK_SIZE,  # 125ms chunks
            )

            log.info("Audio capture initialized (16kHz, mono, 16-bit)")
            return True

        except ImportError:
            log.error("PyAudio not installed. Run: pip install pyaudio")
            return False
        except Exception as e:
            log.error(f"Audio initialization failed: {e}")
            return False

    def _cleanup_audio(self):
        """Clean up audio resources."""
        if self._audio_stream:
            try:
                self._audio_stream.stop_stream()
                self._audio_stream.close()
            except Exception:
                pass
            self._audio_stream = None

        if self._pyaudio:
            try:
                self._pyaudio.terminate()
            except Exception:
                pass
            self._pyaudio = None

    def is_available(self) -> bool:
        """Check if voice control is available (VOSK model exists)"""
        return self._model_path is not None

    def start(self) -> bool:
        """
        Start voice control listening.

        Returns:
            True if started successfully
        """
        if self._running:
            log.warning("Voice control already running")
            return True

        if not self._model_path:
            log.error("Voice control unavailable - no VOSK model")
            return False

        # Initialize VOSK
        if not self._init_vosk():
            play_error_beep()
            return False

        # Initialize audio
        if not self._init_audio():
            play_error_beep()
            return False

        # Start listening thread
        self._running = True
        self.stats["start_time"] = time.time()
        self._thread = threading.Thread(target=self._listen_loop, daemon=True)
        self._thread.start()

        # Play ready beep
        play_ready_beep()

        log.info("Voice control STARTED - Say 'Dictation start' to start, 'Dictation stop' to stop, 'Dictation exit' to close")
        return True

    def stop(self):
        """
        Stop voice control listening and release all resources.

        Terminates the listening thread, closes audio stream, and releases
        VOSK model resources to prevent zombie processes.
        """
        self._running = False

        # Wait for thread to finish
        if self._thread and self._thread.is_alive():
            self._thread.join(timeout=2.0)
        self._thread = None

        # Cleanup audio resources (PyAudio)
        self._cleanup_audio()

        # Release VOSK resources (prevents zombie process)
        try:
            if self._recognizer:
                del self._recognizer
                self._recognizer = None
            if self._model:
                del self._model
                self._model = None
        except Exception as e:
            log.warning(f"VOSK cleanup error: {e}")

        log.info("Voice control STOPPED")

    def pause(self):
        """
        NO-OP: Don't pause voice control during recording.
        We need to keep listening for "stop" command!

        PyAudio (VOSK) and sounddevice (Whisper) can coexist.
        """
        pass  # Keep listening for "stop" command

    def resume(self):
        """
        NO-OP: We don't pause anymore, so nothing to resume.
        """
        pass

    def _listen_loop(self):
        """Main listening loop (runs in background thread)"""
        log.info("Voice control listen loop started")

        while self._running:
            if not self._audio_stream:
                time.sleep(0.1)
                continue

            try:
                # Read audio from microphone
                data = self._audio_stream.read(CHUNK_SIZE, exception_on_overflow=False)

                # Feed to VOSK
                if self._vosk_recognizer.AcceptWaveform(data):
                    # Got a complete phrase - process with alternatives
                    result = self._vosk_recognizer.Result()
                    self._process_result(result)
                else:
                    # Partial result (still speaking)
                    partial = self._vosk_recognizer.PartialResult()
                    self._process_partial(partial)

            except Exception as e:
                if self._running:  # Only log if not shutting down
                    log.error(f"Listen loop error: {e}")
                    time.sleep(0.5)

        log.info("Voice control listen loop ended")

    def _process_result(self, result_json: str):
        """
        Process final recognition result from VOSK.
        Grammar mode gives us exact phrase matches.
        """
        try:
            result = json.loads(result_json)
            text = result.get("text", "").lower().strip()

            if text and text != "[unk]":
                log.info(f"VOSK heard: '{text}'")
                self._check_command(text)

        except json.JSONDecodeError:
            pass

    def _process_partial(self, partial_json: str):
        """
        Process partial result for faster response.

        STRICT MATCHING: Same rules as _check_command:
        1. Max 5 words total
        2. Wake word at START (position 0-1)
        3. Command word immediately after wake word
        """
        try:
            result = json.loads(partial_json)
            text = result.get("partial", "").lower().strip()

            # Skip empty or just [unk]
            if not text or text == "[unk]":
                return

            # Clean leading [unk] tokens (background noise before command)
            clean = self._clean_vosk_text(text)

            # MINIMUM LENGTH after cleaning: "dictation stop" = 14 chars
            if len(clean) < 14:
                return

            # After cleaning, must start with "dictation"
            if not clean.startswith("dictation"):
                return

            log.debug(f"VOSK partial: '{text}' -> clean: '{clean}'")

            # Split cleaned text into words
            words = clean.split()

            # RULE 1: Max 4 words (after cleaning)
            if len(words) > 4 or len(words) < 2:
                return

            # After cleaning, first word must be "dictation"
            if words[0] != "dictation":
                return

            # Command word is at position 1 (right after "dictation")
            cmd_word = words[1]

            # Command word sets
            start_words = {"start"}
            stop_words = {"stop"}
            close_words = {"exit"}
            theme_words = {"dark", "light", "frost"}

            # Only act if command word is valid
            if cmd_word in close_words:
                # EXIT is DESTRUCTIVE - require EXACT match, NO [unk] allowed
                # Partial with [unk] means VOSK was unsure - could be hallucination
                if text.strip().lower() != "dictation exit":
                    log.info(f"PARTIAL EXIT REJECTED (not exact): '{text}'")
                    return
                log.info(f"PARTIAL EXIT: '{text}'")
                self._check_command(text, is_partial=True)
            elif cmd_word in theme_words:
                log.info(f"PARTIAL THEME: '{text}'")
                self._check_command(text, is_partial=True)
            elif cmd_word in stop_words:
                log.info(f"PARTIAL STOP: '{text}'")
                self._check_command(text, is_partial=True)
            elif cmd_word in start_words:
                log.info(f"PARTIAL START: '{text}'")
                self._check_command(text, is_partial=True)

        except json.JSONDecodeError:
            pass

    def _check_command(self, text: str, is_partial: bool = False) -> bool:
        """
        Check if text matches a command and execute callback.

        STRICT MATCHING to prevent false triggers during long dictation:
        1. Text must be EXACTLY 3-4 words (wake + command, with optional [unk] prefix)
        2. Wake word must be at START (position 0-1), not buried in text
        3. Command word must immediately follow wake word
        4. Priority: close > stop > start (check close BEFORE stop)

        Args:
            text: The recognized text
            is_partial: True if this came from partial result (faster but less accurate)
        """
        if not self._validate_command_text(text):
            return False

        now = time.time()
        if self._is_duplicate_or_debounced(text, now):
            return False

        cmd_word = self._extract_command_word(text)
        if cmd_word is None:
            return False

        return self._dispatch_command(cmd_word, text, now)

    def _clean_vosk_text(self, text: str) -> str:
        """Remove leading [unk] tokens - they're just background noise before command."""
        text = text.strip().lower()
        while text.startswith("[unk]"):
            text = text[5:].strip()
        return text

    def _validate_command_text(self, text: str) -> bool:
        """Validate basic text requirements for command matching."""
        if not text or text == "[unk]":
            return False

        # Clean leading [unk] tokens (background noise)
        clean = self._clean_vosk_text(text)

        # After cleaning, must have enough chars for shortest command
        # Shortest valid command: "dictation stop" = 14 chars
        if len(clean) < 14:
            return False

        # After cleaning, must start with "dictation"
        if not clean.startswith("dictation"):
            log.debug(f"Rejected (doesn't start with 'dictation' after cleaning): '{text}'")
            return False

        return True

    def _is_duplicate_or_debounced(self, text: str, now: float) -> bool:
        """Check if command should be skipped due to duplicate or debounce."""
        # DOUBLE-TRIGGER FIX: If exact text was already executed recently, skip
        if text == self._last_executed_text and (now - self._last_executed_time) < 2.0:
            log.debug(f"Skipping duplicate command: '{text}' (already executed)")
            return True
        # Debounce - prevent rapid re-triggers
        if now - self._last_command_time < self._command_cooldown:
            return True
        return False

    def _extract_command_word(self, text: str) -> Optional[str]:
        """Extract and validate command word from text using strict matching rules."""
        # Clean leading [unk] tokens first
        clean = self._clean_vosk_text(text)
        words = clean.split()

        # RULE 1: Maximum 4 words total (after cleaning)
        if len(words) > 4:
            log.debug(f"Rejected (too many words: {len(words)}): '{text}'")
            return None

        # RULE 2: Need at least 2 words: "dictation" + command
        if len(words) < 2:
            return None

        # After cleaning, first word must be "dictation"
        if words[0] != "dictation":
            log.debug(f"Rejected (first word not 'dictation' after cleaning): '{text}'")
            return None

        # Command word is at position 1 (right after "dictation")
        cmd_word = words[1]
        log.info(f"[CMD] text='{text}' | clean='{clean}' | cmd='{cmd_word}'")
        return cmd_word

    def _find_wake_word_position(self, words: list) -> int:
        """Find position of wake word ('dictation') at start of phrase."""
        wake_words = {"dictation"}

        for i in range(min(2, len(words) - 1)):
            if words[i] in wake_words:
                return i
        return -1

    def _dispatch_command(self, cmd_word: str, text: str, now: float) -> bool:
        """Dispatch command to appropriate handler based on command word."""
        # Command word sets
        start_words = {"start"}
        stop_words = {"stop"}
        close_words = {"exit"}
        theme_dark_words = {"dark"}
        theme_light_words = {"light"}
        theme_frost_words = {"frost"}

        # PRIORITY ORDER: exit > themes > off > start
        if cmd_word in close_words:
            # EXIT is DESTRUCTIVE - require EXACT match, NO [unk] allowed
            # This prevents hallucinated commands from closing the app
            text_lower = text.strip().lower()
            if text_lower != "dictation exit":
                log.info(f"EXIT REJECTED (not exact match): '{text}' - must be exactly 'dictation exit'")
                return False
            log.info(f"EXIT command: '{text}'")
            self._mark_executed(text, now)
            self._execute_close()
            return True

        if cmd_word in theme_dark_words:
            log.info(f"THEME DARK command: '{text}'")
            self._mark_executed(text, now)
            self._execute_theme_dark()
            return True

        if cmd_word in theme_light_words:
            log.info(f"THEME LIGHT command: '{text}'")
            self._mark_executed(text, now)
            self._execute_theme_light()
            return True

        if cmd_word in theme_frost_words:
            log.info(f"THEME FROST command: '{text}'")
            self._mark_executed(text, now)
            self._execute_theme_frost()
            return True

        if cmd_word in stop_words:
            log.info(f"OFF command: '{text}'")
            self._mark_executed(text, now)
            self._execute_stop()
            return True

        if cmd_word in start_words:
            log.info(f"START command: '{text}'")
            self._mark_executed(text, now)
            self._execute_start()
            return True

        return False

    def _mark_executed(self, text: str, now: float):
        """Mark command as executed to prevent duplicate triggers."""
        self._last_command_time = now
        self._last_executed_text = text
        self._last_executed_time = now

    def _execute_start(self):
        """Execute START recording command"""
        self.stats["commands_heard"] += 1
        self.stats["start_commands"] += 1

        # Play confirmation beep FIRST (so user knows immediately)
        play_start_beep()

        # Then trigger callback
        if self.on_start_recording:
            try:
                self.on_start_recording()
                log.info("Recording started via voice command")
            except Exception as e:
                log.error(f"Start recording callback failed: {e}")
                play_error_beep()
        else:
            log.warning("No on_start_recording callback set")

    def _execute_stop(self):
        """Execute STOP recording command"""
        self.stats["commands_heard"] += 1
        self.stats["stop_commands"] += 1

        # Play confirmation beep FIRST
        play_stop_beep()

        # Then trigger callback
        if self.on_stop_recording:
            try:
                self.on_stop_recording()
                log.info("Recording stopped via voice command")
            except Exception as e:
                log.error(f"Stop recording callback failed: {e}")
                play_error_beep()
        else:
            log.warning("No on_stop_recording callback set")

    def _execute_close(self):
        """Execute CLOSE app command"""
        self.stats["commands_heard"] += 1
        self.stats["close_commands"] += 1

        # Play confirmation beep FIRST
        play_stop_beep()

        # Then trigger callback
        if self.on_close_app:
            try:
                log.info("App closing via voice command")
                self.on_close_app()
            except Exception as e:
                log.error(f"Close app callback failed: {e}")
                play_error_beep()
        else:
            log.warning("No on_close_app callback set")

    def _execute_theme_dark(self):
        """Execute DARK theme command"""
        self.stats["commands_heard"] += 1
        self.stats["theme_commands"] += 1
        play_start_beep()
        if self.on_theme_dark:
            try:
                self.on_theme_dark()
                log.info("Theme changed to DARK via voice command")
            except Exception as e:
                log.error(f"Theme dark callback failed: {e}")
                play_error_beep()

    def _execute_theme_light(self):
        """Execute LIGHT theme command"""
        self.stats["commands_heard"] += 1
        self.stats["theme_commands"] += 1
        play_start_beep()
        if self.on_theme_light:
            try:
                self.on_theme_light()
                log.info("Theme changed to LIGHT via voice command")
            except Exception as e:
                log.error(f"Theme light callback failed: {e}")
                play_error_beep()

    def _execute_theme_frost(self):
        """Execute FROST theme command"""
        self.stats["commands_heard"] += 1
        self.stats["theme_commands"] += 1
        play_start_beep()
        if self.on_theme_frost:
            try:
                self.on_theme_frost()
                log.info("Theme changed to FROST via voice command")
            except Exception as e:
                log.error(f"Theme frost callback failed: {e}")
                play_error_beep()

    def get_stats(self) -> dict:
        """Get voice control statistics"""
        uptime = 0
        if self.stats["start_time"]:
            uptime = time.time() - self.stats["start_time"]

        return {
            **self.stats,
            "running": self._running,
            "uptime_seconds": uptime,
            "vosk_available": self._vosk_model is not None,
            "model_path": self._model_path,
        }


# ═══════════════════════════════════════════════════════════════
# DEMO / TEST
# ═══════════════════════════════════════════════════════════════

if __name__ == "__main__":
    """Test the voice controller standalone"""

    logging.basicConfig(
        level=logging.DEBUG,
        format="%(asctime)s | %(levelname)s | %(message)s",
        datefmt="%H:%M:%S",
    )

    print("\n" + "=" * 60)
    print("VoicePad Voice Control - GRAMMAR MODE TEST")
    print("=" * 60)
    print("\nGrammar mode enabled - only recognizes specific phrases")
    print("\nCommands (say these):")
    print("  - 'Dictation start' -> Starts recording")
    print("  - 'Dictation stop'  -> Stops recording")
    print("  - 'Dictation exit'  -> Closes app")
    print("\nPress Ctrl+C to exit.\n")

    # Test callbacks
    def test_start():
        print("\n>>> START RECORDING TRIGGERED! <<<\n")

    def test_stop():
        print("\n>>> STOP RECORDING TRIGGERED! <<<\n")

    # Create controller
    controller = VoiceController(
        on_start_recording=test_start,
        on_stop_recording=test_stop,
    )

    if not controller.is_available():
        print("\nVOSK model not found!")
        print("\nDownload from:")
        print("  https://alphacephei.com/vosk/models/vosk-model-small-en-us-0.15.zip")
        print("\nExtract to: models/")
        exit(1)

    # Start listening
    if controller.start():
        try:
            while True:
                time.sleep(1)
        except KeyboardInterrupt:
            print("\n\nGoodbye!")

    controller.stop()
    print("\nStats:", controller.get_stats())
