# -*- coding: utf-8 -*-
"""
VoicePad Beep Confirmation Module

Plays audio beeps so user knows commands worked (even when away from PC).
Uses sounddevice + numpy to generate sine wave tones.
"""

import logging
import threading

log = logging.getLogger("voicepad.voice_control.beep")

# Lazy load numpy and sounddevice
_np = None
_sd = None


def _get_audio_libs():
    """Lazy load numpy and sounddevice."""
    global _np, _sd
    if _np is None:
        import numpy as np
        import sounddevice as sd
        _np = np
        _sd = sd
    return _np, _sd


def _beep_async(frequency: int, duration_ms: int, volume: float = 0.5):
    """Play beep in background thread so it doesn't block."""
    def _play():
        try:
            np, sd = _get_audio_libs()
            duration = duration_ms / 1000.0
            sample_rate = 44100
            t = np.linspace(0, duration, int(sample_rate * duration), dtype=np.float32)
            tone = (volume * np.sin(2 * np.pi * frequency * t)).astype(np.float32)
            sd.play(tone, sample_rate)
            sd.wait()
        except Exception as e:
            log.debug(f"Beep failed: {e}")
    threading.Thread(target=_play, daemon=True).start()


def play_ready_beep():
    """Voice control ready - medium beep"""
    try:
        _beep_async(800, 150, volume=0.4)
        log.info("[BEEP] Ready beep played")
    except Exception as e:
        log.warning(f"Could not play ready beep: {e}")


def play_start_beep():
    """Recording started - high pitched beep"""
    try:
        _beep_async(1000, 150, volume=0.5)
        log.info("[BEEP] Start beep played")
    except Exception as e:
        log.warning(f"Could not play start beep: {e}")


def play_stop_beep():
    """Recording stopped - lower pitched beep"""
    try:
        _beep_async(600, 200, volume=0.5)
        log.info("[BEEP] Stop beep played")
    except Exception as e:
        log.warning(f"Could not play stop beep: {e}")


def play_error_beep():
    """Error - low warning tone"""
    try:
        _beep_async(400, 300, volume=0.6)
        log.info("[BEEP] Error beep played")
    except Exception as e:
        log.warning(f"Could not play error beep: {e}")


def play_wake_beep():
    """Wake word heard - quick high beep"""
    try:
        _beep_async(1200, 100, volume=0.4)
        log.info("[BEEP] Wake beep played")
    except Exception as e:
        log.warning(f"Could not play wake beep: {e}")


# Test function
if __name__ == "__main__":
    import time

    print("Testing VoicePad beeps (Windows system sounds)...")
    print("\n1. Ready beep (SystemExclamation):")
    play_ready_beep()
    time.sleep(1)

    print("\n2. Start beep (SystemAsterisk):")
    play_start_beep()
    time.sleep(1)

    print("\n3. Stop beep (SystemNotification):")
    play_stop_beep()
    time.sleep(1)

    print("\n4. Error beep (SystemHand):")
    play_error_beep()
    time.sleep(1)

    print("\n5. Wake word beep (SystemDefault):")
    play_wake_beep()
    time.sleep(0.5)

    print("\nDone!")
